/* Copyright (c) 2003, 2005, Oracle. All rights reserved.  */
package oracle.i18n.demo.obe.gdkshop.gdk;

import java.util.HashMap;
import java.util.Map;
import oracle.i18n.servlet.ApplicationContext;
import oracle.i18n.servlet.ServletHelper;
import oracle.i18n.servlet.localesource.LocaleSource;

import oracle.i18n.util.OraLocaleInfo;

import java.util.Hashtable;
import java.util.TimeZone;

import javax.naming.Context;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attributes;
import javax.naming.directory.BasicAttribute;
import javax.naming.directory.BasicAttributes;
import javax.naming.directory.DirContext;
import javax.naming.directory.InitialDirContext;
import javax.naming.directory.ModificationItem;
import javax.naming.directory.SearchResult;

import javax.servlet.http.HttpServletRequest;

/**
 * CustomLDAPLocaleSource is a LDAP LocaleSource class that accesses the user
 * locale information on the LDAP server. <code>load</code> method fed demo
 * ls
 * tches  the
 * user locale preference, and <code>store</code> method updates the information.
 * 
 * Default caching : This locale source is cached in the session based on userid 
 */
public class CustomLDAPLocaleSource extends LocaleSource
{
  /** Constant for key to retrieve m_localeMap object */
  private static final String LOCALE_MAP_KEY = "oracle.i18n.demo.obe.gdkshop.gdk.CustomLDAPLocaleSource.m_localeMap";

  // LDAP access driver
  private final static String M_C_LDAP_DRIVER  = 
    "com.sun.jndi.ldap.LdapCtxFactory";

  // LDAP URL
  private static String m_provider_url = "ldap://localhost:389"; // default
  // Root DS
  private static String m_rootds = "dc=example,dc=com";
  // Authentication Method
  private static String m_authentication = "simple";
  // Password for Root DS
  private static String m_credentials = "";

  private final ApplicationContext m_appctx; // GDK application context
  private final DirContext m_initCtx; // LDAP context
  private String m_user;    // user email address
  private String m_user_cn; // user common name

  /**
   * Constructs this object
   *
   * @param userid User object
   * @param appctx GDK ApplicationContext
   *
   * @throws NamingException if the given LDAP name cannot be found.
   */
  private CustomLDAPLocaleSource(String userid, ApplicationContext appctx)
    throws NamingException
  {
    super(true, // cache-able
          false // read-only
    );
    m_user = userid;
    m_appctx = appctx;

    Hashtable env = new Hashtable();
    env.put(Context.INITIAL_CONTEXT_FACTORY, M_C_LDAP_DRIVER);
    env.put(Context.PROVIDER_URL, m_provider_url);
    env.put(Context.SECURITY_AUTHENTICATION, m_authentication);
    env.put(Context.SECURITY_PRINCIPAL, m_rootds);
    env.put(Context.SECURITY_CREDENTIALS, m_credentials);

    m_initCtx = new InitialDirContext(env);
  }

  /**
   * Returns a LocaleSource instance if the user id is valid.
   *
   * @param request HttpServletRequest object
   *
   * @return LocaleSource a LocaleSource object
   *
   * @throws RuntimeException if NamingException occurs
   */
  public static final LocaleSource getInstance(HttpServletRequest request)
  {
    // get authenticated user object
    if (isAccessible(request))
    {
      try
      {
        String userid = (String) request.getSession().getAttribute("userid");
        Map localeMap = (Map)request.getSession().getAttribute(LOCALE_MAP_KEY);
        if (localeMap != null){
            LocaleSource locsrc = (LocaleSource)localeMap.get(userid);
            if (locsrc != null){
                return locsrc;
            }
        }else{   
            localeMap = new HashMap();
        }
        LocaleSource locsrc = new CustomLDAPLocaleSource((String) request.getSession().getAttribute("userid"),
          ServletHelper.getApplicationContextInstance(request));
        locsrc.load();
        localeMap.put(userid, locsrc);
        if (locsrc.isCached()){ 
           request.getSession().setAttribute(LOCALE_MAP_KEY, localeMap);
        }
        return locsrc;     
          
      }
      catch (NamingException e)
      {
        throw new RuntimeException("NamingException: " + e.getExplanation());
      }
    }

    return null;
  }

  /**
   * Returns true is LocaleSource is accessible
   *
   * @param request HttpServletRequest object
   *
   * @return boolean
   */
  public static boolean isAccessible(HttpServletRequest request)
  {
    String userid = (String) request.getSession().getAttribute("userid");

    return userid != null;
  }

  /**
   * Fetches all locale information from LDAP server.
   *
   * @return true if the operation is success, otherwise false
   */
  public boolean load()
  {
    Attributes matchAttrs = new BasicAttributes(true); // ignore case
    matchAttrs.put(new BasicAttribute("mail", m_user));
    m_user_cn = null;

    try 
    {
      NamingEnumeration answer = m_initCtx.search(m_rootds, matchAttrs);

      if (answer.hasMore())
      {
        SearchResult sr = (SearchResult) answer.next();
        m_user_cn = sr.getName();

        Attributes attrs = sr.getAttributes();
        String plang = (String) attrs.get("preferredLanguage").get();
        String tzs = (String) attrs.get("timeZone").get();
        setLocale(OraLocaleInfo.getLocaleFromString(plang));
        setTimeZone(TimeZone.getTimeZone(tzs));
      }
      else
      {
        m_user = null;
      }

      answer.close();
    }
    catch (NamingException e)
    {
      m_user = null;
    }

    return m_user != null;
  }

  /**
   * Stores all locale information into the LDAP server.
   *
   * @return true if the operation is success, otherwise false
   */
  public boolean store()
  {
    if (!load()) // make sure m_user_cn is set
    {
      return false;
    }

    // updates 
    try
    {
      ModificationItem[] mods = new ModificationItem[2];
      String plang = getLocale().toString().replace('_', '-');
      mods[0] = new ModificationItem(DirContext.REPLACE_ATTRIBUTE,
          new BasicAttribute("preferredLanguage", plang));

      String tzs = getTimeZone().getID();
      mods[1] = new ModificationItem(DirContext.REPLACE_ATTRIBUTE,
          new BasicAttribute("timeZone", tzs));
      m_initCtx.modifyAttributes(m_user_cn + m_rootds, mods);
    }
    catch (NamingException e)
    {
      m_user = null;
    }

    return m_user != null;
  }

  // initializes the LDAP connection
  public static synchronized void setLdapConnection(
    String ldapcon, String rootds, 
    String authentication, String credentials)
  {
    // LDAP connection URL
    m_provider_url = (ldapcon != null ? ldapcon : "");

    // root DS
    m_rootds = (rootds != null ? rootds : "");
    
    // Authentication Method
    m_authentication = (authentication != null ? authentication : "");

    // Password for Root DS
    m_credentials = (credentials != null ? credentials : "");
  }
}

